<?php
/**
 * @file
 * Theme functions for the Simpletest module.
 */

/**
 * Returns HTML for a test list generated by simpletest_test_form() into a table.
 *
 * @param $variables
 *   An associative array containing:
 *   - table: A render element representing the table.
 *
 * @ingroup themeable
 */
function theme_simpletest_test_table($variables) {
  $table = $variables['table'];

  backdrop_add_css(backdrop_get_path('module', 'simpletest') . '/css/simpletest.css');
  backdrop_add_js(backdrop_get_path('module', 'simpletest') . '/js/simpletest.js');
  backdrop_add_js('core/misc/tableselect.js');

  // Create header for test selection table.
  $header = array(
    array('class' => array('select-all')),
    array('data' => t('Test'), 'class' => array('simpletest-test')),
    array('data' => t('Description'), 'class' => array('simpletest_description')),
  );

  // Define the images used to expand/collapse the test groups.
  $js = array(
    'images' => array(
      theme('image', array('path' => 'core/misc/menu-collapsed.png', 'width' => 7, 'height' => 7, 'alt' => t('Expand'), 'title' => t('Expand'))) . ' <a href="#" class="simpletest-collapse">(' . t('Expand') . ')</a>',
      theme('image', array('path' => 'core/misc/menu-expanded.png', 'width' => 7, 'height' => 7, 'alt' => t('Collapse'), 'title' => t('Collapse'))) . ' <a href="#" class="simpletest-collapse">(' . t('Collapse') . ')</a>',
    ),
  );
  backdrop_add_js(array('simpleTest' => $js), 'setting');

  // Cycle through each test group and create a row.
  $rows = array();
  foreach (element_children($table) as $key) {
    $element = &$table[$key];
    $row = array();

    // Make the class name safe for output on the page by replacing all
    // non-word/decimal characters with a dash (-).
    $test_class = strtolower(trim(preg_replace("/[^\w\d]/", "-", $key)));

    // Select the right "expand"/"collapse" image, depending on whether the
    // category is expanded (at least one test selected) or not.
    $collapsed = !empty($element['#collapsed']);
    $image_index = $collapsed ? 0 : 1;

    // Place-holder for checkboxes to select group of tests.
    $row[] = array('id' => $test_class, 'class' => array('simpletest-group-select-all'));

    // Expand/collapse image and group title.
    $row[] = array(
      'data' => '<span class="simpletest-image" id="simpletest-test-group-' . $test_class . '"></span>' .
        '<label for="' . $test_class . '-group-select-all">' . $key . '</label>',
      'class' => array('simpletest-group-label'),
    );

    $row[] = array(
      'data' => '&nbsp;',
      'class' => array('simpletest-group-description'),
    );

    $rows[] = array('data' => $row, 'class' => array('simpletest-group'));

    // Sorting $element by children's #title attribute instead of by class name.
    backdrop_sort($element, array('#title' => SORT_STRING));

    // Cycle through each test within the current group.
    foreach (element_children($element) as $test_name) {
      $test = $element[$test_name];
      $row = array();

      // Store test title and description so that checkbox won't render them.
      $title = $test['#title'];
      $description = $test['#description'];

      $test['#title_display'] = 'invisible';
      unset($test['#description']);

      $row[] = array(
        'data' => backdrop_render($test),
        'class' => array('simpletest-test-select'),
      );
      $row[] = array(
        'data' => '<label for="' . $test['#id'] . '">' . $title . '</label>',
        'class' => array('simpletest-test-label'),
      );
      $row[] = array(
        'data' => '<div class="description">' . $description . '</div>',
        'class' => array('simpletest-test-description'),
      );

      $rows[] = array('data' => $row, 'class' => array($test_class . '-test', ($collapsed ? 'js-hide' : '')));
    }
    unset($table[$key]);
  }

  if (empty($rows)) {
    return '<strong>' . t('No tests to display.') . '</strong>';
  }
  else {
    return theme('table', array('header' => $header, 'rows' => $rows, 'attributes' => array('id' => 'simpletest-form-table')));
  }
}

/**
 * Returns HTML for the summary status of a simpletest result.
 *
 * @param $variables
 *   An associative array containing:
 *   - form: A render element representing the form.
 *
 * @ingroup themeable
 */
function theme_simpletest_result_summary($variables) {
  $form = $variables['form'];
  return '<div class="simpletest-' . ($form['#ok'] ? 'pass' : 'fail') . '">' . _simpletest_format_summary_line($form) . '</div>';
}

